<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Usertag;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\Post\PostUserTagTable;
use Erlage\Photogram\Data\Models\Post\Usertag\PostUserTagModel;

final class PostUserTagContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(
                PostUserTagTable::TAGGED_USER_ID,
                RequestTable::PAYLOAD,
                PostUserTagTable::TABLE_NAME
            );

            $offsetFromReq = self::$request -> findKeyOffset(PostUserTagTable::ID, PostUserTagTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            |  ensure target model exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            |  privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($targetUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::PRIVATE_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $postUserTagTableQuery = (new Query()) -> from(PostUserTagTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $postUserTagTableQuery -> greaterThan(PostUserTagTable::ID, $offsetFromReq);
            }
            else
            {
                $postUserTagTableQuery -> lessThan(PostUserTagTable::ID, $offsetFromReq);
            }

            /*
            |--------------------------------------------------------------------------
            | where user is tagged
            |--------------------------------------------------------------------------
            */

            $postUserTagTableQuery -> where(PostUserTagTable::TAGGED_USER_ID, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $postUserTagTableQuery
                -> orderByDesc(PostUserTagTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_POST_USER_TAG));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $postUserTagBeans = $postUserTagTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($postUserTagBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | process beans
            |--------------------------------------------------------------------------
            */

            self::processBeans(PostUserTagTable::getTableName(), $postUserTagBeans, function (PostUserTagModel $postUserTagModel)
            {
                self::addDependency(PostTable::getTableName(), $postUserTagModel -> getTaggedInPostId());

                self::addDependency(UserTable::getTableName(), $postUserTagModel -> getPostOwnerUserId());
            });

            /*
            |--------------------------------------------------------------------------
            | process dependencies
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*
            |--------------------------------------------------------------------------
            | additional data | post likes maps
            | -------------------------------------------------------------------------
            | help build like button
            |--------------------------------------------------------------------------
            */

            $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                PostLikeTable::getTableName(),
                array(
                    PostLikeTable::LIKED_POST_ID    => $postsContainer -> getIds(),
                    PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | additional data | post save maps
            | -------------------------------------------------------------------------
            | help build post save button
            |--------------------------------------------------------------------------
            */

            self::fetchModelsAndAddAsAdditional(
                PostSaveTable::getTableName(),
                array(
                    PostSaveTable::SAVED_POST_ID    => $postsContainer -> getIds(),
                    PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }
}
